#!/usr/bin/env python3
"""
plot_adjoint_ratio.py
=====================

Read the results of an adjoint volume sweep and visualise the ratio of
adjoint string tensions between SU(3) and SU(2) gauge groups as a
function of the lattice size ``L``.  The script computes the ratio

    R(L) = mean_{b,k,n0}(σ_adj_SU3(L)) / mean_{b,k,n0}(σ_adj_SU2(L))

along with the standard deviation across the parameter grid.  The
resulting plot overlays a horizontal reference line at the theoretical
Casimir ratio to facilitate comparison.

Example usage::

    python plot_adjoint_ratio.py --input data/results/vol4_wilson_loop_adjoint_volume_sweep/adjoint_volume_summary.csv --output plots/adjoint_ratio.png

"""

import argparse
import os
import pandas as pd
import matplotlib.pyplot as plt


def main() -> None:
    parser = argparse.ArgumentParser(description="Plot SU3/SU2 adjoint tension ratio vs lattice size")
    parser.add_argument("--input", required=True, help="Path to adjoint_volume_summary.csv")
    parser.add_argument("--output", required=True, help="Path to save the PNG plot")
    args = parser.parse_args()

    df = pd.read_csv(args.input)
    # Group by gauge, aggregate mean across parameter grid
    # Only use adjoint tensions (sigma_adj)
    grouped = df.groupby(["b", "k", "n0", "L", "gauge"]).agg({"sigma_adj": "mean"}).reset_index()
    # Separate SU2 and SU3
    df_su2 = grouped[grouped["gauge"] == "SU2"][["b", "k", "n0", "L", "sigma_adj"]].rename(columns={"sigma_adj": "sigma_adj_SU2"})
    df_su3 = grouped[grouped["gauge"] == "SU3"][["b", "k", "n0", "L", "sigma_adj"]].rename(columns={"sigma_adj": "sigma_adj_SU3"})
    # Merge on parameter combinations
    merged = pd.merge(df_su3, df_su2, on=["b", "k", "n0", "L"], how="inner")
    # Compute ratio for each combination
    merged["R"] = merged["sigma_adj_SU3"] / merged["sigma_adj_SU2"]
    # Aggregate by L
    stats = merged.groupby("L")["R"].agg(["mean", "std"]).reset_index()

    # Plot
    plt.figure(figsize=(6, 4))
    plt.errorbar(stats["L"], stats["mean"], yerr=stats["std"], marker="o", linestyle="-", label="Measured R")
    # Theoretical Casimir ratio for SU3 vs SU2: (2*9)/(9-1) / ((2*4)/(4-1)) = (18/8) / (8/3) ??? Wait compute maybe 1.125? no: actual ratio is (σ_adj_SU3/σ_fund_SU3)/(σ_adj_SU2/σ_fund_SU2) = ratio of Casimir factors: ((2*9)/(9-1))/((2*4)/(4-1)) = (18/8)/(8/3) = (2.25)/(2.666...) = 0.84375. We'll compute explicitly below.
    # Compute theoretical ratio
    casimir_su2 = (2 * 4) / (4 - 1)  # SU2 Casimir factor
    casimir_su3 = (2 * 9) / (9 - 1)  # SU3 Casimir factor
    theoretical_ratio = casimir_su3 / casimir_su2
    plt.hlines(theoretical_ratio, xmin=stats["L"].min(), xmax=stats["L"].max(), linestyle="--", label=f"Casimir ratio={theoretical_ratio:.3f}")
    plt.xlabel("Lattice size L")
    plt.ylabel("σ_adj_SU3 / σ_adj_SU2")
    plt.title("Adjoint tension ratio SU3/SU2 vs L")
    plt.xticks(stats["L"])
    plt.legend()
    plt.tight_layout()
    # Ensure output directory exists
    os.makedirs(os.path.dirname(args.output), exist_ok=True)
    plt.savefig(args.output)
    plt.close()
    print(f"Saved plot to {args.output}")


if __name__ == "__main__":  # pragma: no cover
    main()